// ============================================================================
// ============================================================================
// ============================================================================
// ==                                                                        ==
// == Name    : TheEmuLib.Emu_Ripple_RGBA_XY.fsh                             ==
// == Type    : Fragment shader                                              ==
// == Version : 1.1.4 (2017/01/`5)                                           ==
// == Creator : TheEmu © TheEmu 2017, Some Rights Reserved                   ==
// == Licence : Creative Commons Attribution-ShareAlike 4.0                  ==
// ==           http://creativecommons.org/licences/by-sa/4.0                ==
// ==                                                                        ==
// == Purpose: To apply a "ripple" to the colours and opacities of points in ==
// == an image with the ripple at a point a function of  the  point's  (x,y) ==
// == coordinates.                                                           ==
// ==                                                                        ==
// == Description: A source image is manipulated by wobbling its colours and ==
// == its opacities.  The  ripple at at any point comprises the sum of up to ==
// == four terms added to its colour components, each term being of the form ==
// ==                                                                        ==
// ==        K0 * A * cos ( T*F - B*x - C*y + D ) + K1                       ==
// ==                                                                        ==
// == where T is time, (x,y) the normalised coordinates and A, B, C, D, F, K ==
// == are ripple parameters supplied as shader inputs.  The negation of some ==
// == terms involving B, C, D and K is done so that positive values of these ==
// == parameters correspond to movement in a positive direction.             ==
// ==                                                                        ==
// == Static distortion of the source image is obtained using a frequency of ==
// == of 0.0 for all components of the ripple.                               ==
// ==                                                                        ==
// == This file is a member of The Emu's shader library.                     ==
// ==                                                                        ==
// == ====================================================================== ==
// ==                                                                        ==
// == Update history:                                                        ==
// ==                                                                        ==
// ==   2016/12/15 - v1.0.0 - Initial version, from Emu_Ripple_XY.fsh        ==
// ==   2016/12/16 - v1.0.1 - Regularised format of TheEmuLib sources.       ==
// ==   2016/12/17 - v1.0.2 - Phases now in cycles for consistancy.          ==
// ==   2016/12/17 - v1.0.3 - Borders replaced by edge coordinates.          ==
// ==   2016/12/18 - v1.0.4 - Added edge_drops to drop the unrippled region. ==
// ==   2016/12/19 - v1.0.5 - Eliminated unused, for RGBA, edge hold cases.  ==
// ==   2016/12/20 - v1.0.6 - Added optional RGBA clamping.                  ==
// ==   2016/12/20 - v1.0.7 - Added overall ripple scale and offset.         ==
// ==   2016/12/21 - v1.0.8 - Changed signs of B and K terms.                ==
// ==   2016/12/21 - v1.1.0 - Changed name to have _XY as a suffix.          ==
// ==   2017/01/05 - v1.1.1 - Renamed all Wobble shaders to be Ripple.       ==
// ==   2017/01/06 - v1.1.2 - mix(vec4,vec4,bvec4) not allowed by NVIDIA.    ==
// ==   2017/01/15 - v1.1.3 - Fixed bug in edge drop logic.                  ==
// ==   2017/01/15 - v1.1.4 - Added support for scale and hotspot.           ==
// ==                                                                        ==
// ============================================================================
// ============================================================================
// ============================================================================

// ============================================================================
// == Standard shader inputs ==================================================
// ============================================================================

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// The image that is to be manipulated.

uniform sampler2D iChannel0;

// ============================================================================
// == Imports from TheEmuLib ==================================================
// ============================================================================
//
// The GLSL shader language currently provides no mechanism for importing  any
// elements that are defined in other modules, not even C's crude source level
// #include mechanism. In the absence of anything better TheEmuLib handles any
// imports by manualy copying relevent utility code snippets from the  sources
// in the Shader Lib.Inc directory. This is very crude but I have attempted to
// be systematic in the way in which this is presented in the library sources.
//
// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Basic_Constants.lib.src

#define PI  3.1415926535897932384626433832795
#define TAU 6.2831853071795864769252867665590

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Common_Utilities.lib.src

#define EMU_DEFAULT(type,x,default_value) ( (x==type(0.0)) ? (default_value) : (x) )

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Functions from TheEmuLib.Emu_Coordinate_Normalisation.lib.src

#define EMU_NORMALISE_TO_WINDOW_1(xy,wsize) (     (xy)/(wsize) )
#define EMU_NORMALISE_TO_WINDOW_2(xy,wsize) ( 2.0*(xy)/(wsize) - 1.0 )

#define EMU_NORMALISE_TO_WINDOW_3(xy,c1,c2) ( EMU_NORMALISE_TO_WINDOW_1(xy-c1,c2-c1) )
#define EMU_NORMALISE_TO_WINDOW_4(xy,c1,c2) ( EMU_NORMALISE_TO_WINDOW_2(xy-c1,c2-c1) )

#define EMU_DENORMALISE_TO_WINDOW_1(uv,size)  (      (uv)        * (size) )
#define EMU_DENORMALISE_TO_WINDOW_2(uv,size)  ( ( (uv)*0.5+0.5 ) * (size) )

#define EMU_DENORMALISE_TO_WINDOW_3(uv,c1,c2) ( EMU_DENORMALISE_TO_WINDOW_1(uv,c2-c1)+c1 )
#define EMU_DENORMALISE_TO_WINDOW_4(uv,c1,c2) ( EMU_DENORMALISE_TO_WINDOW_2(uv,c2-c1)+c1 )

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

vec2 Emu_Normalise_to_Window ( vec2 xy )
 { return EMU_NORMALISE_TO_WINDOW_1 ( xy, u_WindowSize.xy );
 }

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

vec2 Emu_Normalise_to_Window ( vec2 xy, vec2 xy0, vec2 xy1 )
 { return EMU_NORMALISE_TO_WINDOW_3 ( xy, xy0, xy1 );
 }

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

vec2 Emu_Denormalise_to_Window ( vec2 uv, vec2 xy0, vec2 xy1 )
 { return EMU_DENORMALISE_TO_WINDOW_3 ( uv, xy0, xy1 );
 }

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Boolean_Vector_Ops.lib.src

#define Emu_bvec4_or(A,B)  bvec4 ( A[0]||B[0], A[1]||B[1], A[2]||B[2], A[3]||B[3] )
#define Emu_bvec4_and(A,B) bvec4 ( A[0]&&B[0], A[1]&&B[1], A[2]&&B[2], A[3]&&B[3] )
#define Emu_bvec4_xor(A,B) bvec4 ( A[0]^^B[0], A[1]^^B[1], A[2]^^B[2], A[3]^^B[3] )

// ============================================================================
// == Shader specific inputs ==================================================
// ============================================================================

uniform vec2 Emu_Ripple_XY_scale;
uniform vec2 Emu_Ripple_XY_hotspot;

vec2 scale   = EMU_DEFAULT ( vec2, Emu_Ripple_XY_scale,   vec2(1.0) );
vec2 hotspot = EMU_DEFAULT ( vec2, Emu_Ripple_XY_hotspot, vec2(0.0) );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// By default the whole area of the source image is rippled, but this can be
// overridden by using the following parameter to limit the to wobbling to a
// particular region.  The  edges parameter defines the region that is to be
// rippled by specifying the low and high limits of normalised coordinates.

uniform vec4 Emu_Ripple_RGBA_XY_edges;

vec4 edges =
  EMU_DEFAULT ( vec4, Emu_Ripple_RGBA_XY_edges, vec4(0.0,1.0,0.0,1.0) );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The effect of the wobbling at the edges of the rippled area can result in
// discontinuities in the generated image.  By default any tears between the
// rippled area and its suroundings represented by fully transparent  black,
// i.e. they become holes through which any background image can be seen.
//
// The default behaviour of producing tears at the edges can be overidden by
// using the edge_mode parameter, which may have the values:-
//
//   0 - edge_tears - Default, produces "tears" at the edge
//   1 - edge_hold1 - Holds ripples at edges of rippled region to 0.0
//
// Note, the edge modes are a subset of those available in the Ripple_XY and
// and Ripple_RA shaders and are named accordingly even though  the  concept
// of a tear is not really applicable here except in a very abstract sense.
//
// In mode 0 there is a discontinity at a tear.
// In mode 1 ripples fade out near the edges.
//
// If 10 is added to the edge mode then anything outside the rippled area is
// rendered transparent rather than retaining its original colour.
//
// The edge mode is specified separately for all four edges using  the  vec4
// input parameter Emu_Ripple_RGBA_XY_edge_mode where the elements give  the
// mode for the edges at X=0, X=1, Y=0 and Y=1 in that order.  Unfortunately
// the iStripper program does not support passing an ivec4 from a scene file
// so we have to use a vec4 and ignore any fractional part.

uniform vec4 Emu_Ripple_RGBA_XY_edge_mode;

ivec4 edge_mode1 = ivec4 ( mod(Emu_Ripple_RGBA_XY_edge_mode/10.0,10.0) );
ivec4 edge_mode2 = ivec4 ( mod(Emu_Ripple_RGBA_XY_edge_mode,10.0) );

bvec4 edge_drops = equal ( edge_mode1, ivec4(1) );

bvec4 edge_undef = greaterThan ( edge_mode2, ivec4(1) );

bvec4 edge_hold1 =                equal(edge_mode2,ivec4(1));
bvec4 edge_tears = Emu_bvec4_or ( equal(edge_mode2,ivec4(0)), edge_undef );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// By default no clamping of the RGBA colour components is done, which means
// that they can very easily go out of their valid range of 0.0 to 1.0 which
// can sometimes be a problem - typically leading to areas of flat colour in
// the output image.  The clamp_mode parameter allows some control over this
// behaviour by performing hard or soft caming in this shader. The following
// values may be used for this parameter:-
//
//    0 - No clamping (default)
//    1 - Hard clamping
//    2 - Soft clamping

uniform vec4 Emu_Ripple_RGBA_XY_clamp_mode;

ivec4 clamp_mode = ivec4 ( Emu_Ripple_RGBA_XY_clamp_mode );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// Overall scale and offset for the ripple, the K0 and K1 parameters refered
// to in the initial description of the shader. K0 is not strictly necessary
// as the same effect could be obtained in the scene file by scaling each of
// the A terms, but it is convenient for the shader user to be able to tweak
// the overall scale rather than changing all of the A parameters.  K1  just
// adds a constant offset to the ripple,  the  same effect could be obtained
// in the scene file by altering the position of the object being rippled.

uniform vec2 Emu_Ripple_RGBA_XY_KK;  // Ripple overall scale and offset

vec2 KK = EMU_DEFAULT ( vec2, Emu_Ripple_RGBA_XY_KK, vec2(1.0,0.0) );

// The ripple parameters for each component of the ripple. Any parameter not
// specified defaults to zero. The spatial frequency terms are the number of
// ripples per unit distance.

uniform vec4 Emu_Ripple_RGBA_XY_Ar;  // Amplitudes
uniform vec4 Emu_Ripple_RGBA_XY_Ag;
uniform vec4 Emu_Ripple_RGBA_XY_Ab;
uniform vec4 Emu_Ripple_RGBA_XY_Aa;

uniform vec4 Emu_Ripple_RGBA_XY_Br;  // Spatial frequencies (X component terms)
uniform vec4 Emu_Ripple_RGBA_XY_Bg;
uniform vec4 Emu_Ripple_RGBA_XY_Bb;
uniform vec4 Emu_Ripple_RGBA_XY_Ba;

uniform vec4 Emu_Ripple_RGBA_XY_Cr;  // Spatial frequencies (Y component terms)
uniform vec4 Emu_Ripple_RGBA_XY_Cg;
uniform vec4 Emu_Ripple_RGBA_XY_Cb;
uniform vec4 Emu_Ripple_RGBA_XY_Ca;

uniform vec4 Emu_Ripple_RGBA_XY_Dr;  // Phases in fractions of a cycle
uniform vec4 Emu_Ripple_RGBA_XY_Dg;
uniform vec4 Emu_Ripple_RGBA_XY_Db;
uniform vec4 Emu_Ripple_RGBA_XY_Da;

uniform vec4 Emu_Ripple_RGBA_XY_Fr;  // Frequencies in Hz
uniform vec4 Emu_Ripple_RGBA_XY_Fg;
uniform vec4 Emu_Ripple_RGBA_XY_Fb;
uniform vec4 Emu_Ripple_RGBA_XY_Fa;

// The ripple parameters as used in the body of the shader.

#define Ar ( Emu_Ripple_RGBA_XY_Ar )
#define Ag ( Emu_Ripple_RGBA_XY_Ag )
#define Ab ( Emu_Ripple_RGBA_XY_Ab )
#define Aa ( Emu_Ripple_RGBA_XY_Aa )

#define Br ( Emu_Ripple_RGBA_XY_Br * TAU )
#define Bg ( Emu_Ripple_RGBA_XY_Bg * TAU )
#define Bb ( Emu_Ripple_RGBA_XY_Bb * TAU )
#define Ba ( Emu_Ripple_RGBA_XY_Ba * TAU )

#define Cr ( Emu_Ripple_RGBA_XY_Cr * TAU )
#define Cg ( Emu_Ripple_RGBA_XY_Cg * TAU )
#define Cb ( Emu_Ripple_RGBA_XY_Cb * TAU )
#define Ca ( Emu_Ripple_RGBA_XY_Ca * TAU )

#define Dr ( Emu_Ripple_RGBA_XY_Dr * TAU )
#define Dg ( Emu_Ripple_RGBA_XY_Dg * TAU )
#define Db ( Emu_Ripple_RGBA_XY_Db * TAU )
#define Da ( Emu_Ripple_RGBA_XY_Da * TAU )

#define Fr ( Emu_Ripple_RGBA_XY_Fr * TAU )
#define Fg ( Emu_Ripple_RGBA_XY_Fg * TAU )
#define Fb ( Emu_Ripple_RGBA_XY_Fb * TAU )
#define Fa ( Emu_Ripple_RGBA_XY_Fa * TAU )

// ============================================================================
// Shader specific utility functions ==========================================
// ============================================================================

// The clamp function like macros are used simply to make the code where they
// are used neater and clearer. Each macro adds ripples, w, to the colour, c,
// subject to different forms of clamping.
//
//   clamp_0 - no clamping
//   clamp_1 - hard clamping to range 0.0 to 1.0
//   clamp_2 - soft clamping to range 0.0 to 1.0

#define clamp_0(c,w) ( c + w )
#define clamp_1(c,w) clamp ( c+w, 0.0, 1.0 )
#define clamp_2(c,w) smoothstep ( 0.0, -w, c ) * ( 1.0-smoothstep ( 0, w, 1.0-c ) )

// ============================================================================
// The shader's major function ================================================
// ============================================================================

vec4 Emu_Ripple_RGBA_XY ( vec2 uv0 )
 {
   // Normalise to the region that is to be rippled.

   vec2 pq = Emu_Normalise_to_Window ( uv0, edges.xz, edges.yw );

   // Determine which edges affect the ripple at this point.

   bvec4 edge = bvec4 ( pq.x < 0.5, pq.x > 0.5,
                        pq.y < 0.5, pq.y > 0.5
                      );

   // Update selected edge flags to reflect the nearest edges.

   edge_drops = Emu_bvec4_and ( edge, edge_drops );

   // If outside the rippled area then no ripple is applied and we just have
   // to return the colour corresponding to the unrippled coordinates unless
   // the outer region is to be dropped, i.e. rendered transparent.

   if ( any ( notEqual(pq,fract(pq)) ) )
    { bool drop = any(edge_drops);
      return mix ( texture2D(iChannel0,uv0), vec4(0.0), vec4(drop) );
    }

   // Update the remaining edge flags to reflect the nearest edges.

   edge_tears = Emu_bvec4_and ( edge, edge_tears );
   edge_hold1 = Emu_bvec4_and ( edge, edge_hold1 );

   // Calculate the four components for both the x and y ripples.
   // See comments at top of shader for why B and C are negated.

   vec4 wr = Ar * cos ( u_Elapsed*Fr - Br*pq.x - Cr*pq.y + Dr );
   vec4 wg = Ag * cos ( u_Elapsed*Fg - Bg*pq.x - Cg*pq.y + Dg );
   vec4 wb = Ab * cos ( u_Elapsed*Fb - Bb*pq.x - Cb*pq.y + Db );
   vec4 wa = Aa * cos ( u_Elapsed*Fa - Ba*pq.x - Ca*pq.y + Da );

   // Apply overall ripple scale and offset parameters.

   wr = KK[0]*wr + KK[1];
   wg = KK[0]*wg + KK[1];
   wb = KK[0]*wb + KK[1];
   wa = KK[0]*wa + KK[1];

   // Sum the four components of the wobbling for each colour.

   vec4 ww = vec4 ( dot ( vec4(1.0), wr ),
                    dot ( vec4(1.0), wg ),
                    dot ( vec4(1.0), wb ),
                    dot ( vec4(1.0), wa )
                  );

   // Optionaly force the ripples to to be zero at the edges. This is done
   // by multiplying them by a function, sin, that is zero at the edges.

   vec2 ff = sin(pq*PI);

   if ( edge_hold1[0] || edge_hold1[1] ) ww *= ff.x;
   if ( edge_hold1[2] || edge_hold1[3] ) ww *= ff.y;

   // Renormalise to the dimensions of the whole image.

   vec2 st = Emu_Denormalise_to_Window ( pq, edges.xz, edges.yw );

   // Get the colour for the pixel.

   vec4 colour = texture2D ( iChannel0, fract(st) );

   // Apply the ripple.

   switch ( clamp_mode.r ) {
      case 0: colour.r = clamp_0 ( colour.r, ww.r ); break;
      case 1: colour.r = clamp_1 ( colour.r, ww.r ); break;
      case 2: colour.r = clamp_1 ( colour.r, ww.r ); break;
   }

   switch ( clamp_mode.g ) {
      case 0: colour.g = clamp_0 ( colour.g, ww.g ); break;
      case 1: colour.g = clamp_1 ( colour.g, ww.g ); break;
      case 2: colour.g = clamp_1 ( colour.g, ww.g ); break;
   }

   switch ( clamp_mode.b ) {
      case 0: colour.b = clamp_0 ( colour.b, ww.b ); break;
      case 1: colour.b = clamp_1 ( colour.b, ww.b ); break;
      case 2: colour.b = clamp_1 ( colour.b, ww.b ); break;
   }

   switch ( clamp_mode.a ) {
      case 0: colour.a = clamp_0 ( colour.a, ww.a ); break;
      case 1: colour.a = clamp_1 ( colour.a, ww.a ); break;
      case 2: colour.a = clamp_1 ( colour.a, ww.a ); break;
   }

   return colour;

 }

// ============================================================================
// == The shader's main routine ===============================================
// ============================================================================

void main ( void )
 {
   // Get the normalised coordinates of the current point.

   vec2 uv = Emu_Normalise_to_Window ( gl_FragCoord.xy );

   uv = ( uv - hotspot ) / scale + hotspot;

   // Apply the ripple and update the shader's outputs.

   gl_FragColor = Emu_Ripple_RGBA_XY(uv) * gl_Color;

 }

// ============================================================================
